/* 'Arrows' Game written by xexyl (from https://share.xexyl.net/games/arrows)
 *
 * The object of the game is thus:
 *    
 *  The game prints out five evenly spaced arrows in the centre of the screen.
 *  The idea is that the player has up to DELAY seconds to use their left or
 *  right arrow key depending on the direction the middle arrow is facing. Uses
 *  non-blocking I/O and if after DELAY seconds there is still no input the game
 *  will proceed; in this case the score stays the same but the loops counter is
 *  increased (ironically decreasing the total score at the end! :). Hit 'q' to
 *  quit the game at which point your score (out of total loops) will be printed
 *  along with the percentage you got correct.
 *
 * To compile make sure the Makefile, arrows.h (and obviously this file) are in
 * the same directory; then type:
 *  
 *	make
 *
 * If you want to change the number of seconds to delay see the COMPILING file.
 *
 * Run with:
 *	
 *	./arrows
 *
 * Commands:
 *
 *	Q = quit
 * 
 * Use left and right arrows on keyboard to choose direction the middle arrow is
 * facing. Scoring won't start until you make you first valid move (right or
 * left arrow key).
 * 
 *
 * Version 4 - 2/July/2020
 * 
 * (Amusingly I made version 3 the same day of the year as version 2
 * only 6 years later; Version 2 was 22/April/2013 and version 3 was
 * 22/April/2019)
 *
 * This is in the public domain just please keep it as it is.
 *
 * - xexyl
 */
#include <curses.h>
#include <time.h>
#include "arrows.h"

static unsigned long
prng()
{
    register unsigned long lo, hi, test;

    lo = hi = test = 0;

    if (q)
    	hi = seed / q;

    lo = seed % q;

    test = a * lo - r * hi;

    if (test > 0)
	seed = test;
    else
	seed = test + m;

    return seed;
}

static unsigned long
number(unsigned long from, unsigned long to)
{
    return (prng() % to - from) - 1;
}
static void
update_screen()
{
    static int row, col;

    attron(A_BOLD);
    erase();
    
    getmaxyx(stdscr, row, col);
    move(row/2, (col - LENGTH)/2);
}
static void
init_screen()
{
    initscr();

    update_screen();
}
/* Allow keypad (arrows etc.) and hide cursor */
static void
init_keyboard()
{
    keypad(stdscr, TRUE);
    curs_set(0);
}

static void
setup()
{
    seed = (unsigned long) time(NULL);
    printf("%s", instructions);

    getc(stdin);
}

static void
print_spaces()
{
    printw("      ");   
}

static void
display_score(unsigned long long score, unsigned long long loops)
{
    double ratio = (((double)score/(double)loops) * 100.0);

    endwin();

    printf("Your score was %llu out of %llu - %.2f percent correct.\n", score, loops, ratio);
}

static void
play()
{
    int ch, start = 0, i;
    unsigned direction, done;
    unsigned long long score, loops;

    done = 0;
    score = loops = 0;

    /* XXX initscr() has to be called before keypad() so the order here actually
     * matters just as an aside (this caused me an issue when I was making this
     * more modular and also adding a proper pRNG (though still some to be
     * desired).
     */
    init_screen();
    init_keyboard();

    while (!done) 
    {
	update_screen();

	direction = NONE;
	for (i = 0; i < 5; i++)
	{
	    unsigned long num;

	    print_spaces();
	    if ((num = number(0, 10000)) < 5000)
	    {
		addch(ACS_LARROW);
		
		if (i == 2) direction = LEFT;
	    }
	    else 
	    {
		addch(ACS_RARROW);
		
		if (i == 2) direction = RIGHT;
	    }
	}
      
	refresh();
	timeout(DELAY_MS);
	
	ch = getch();
	switch (ch)
	{
	    case KEY_LEFT:
		if (direction == LEFT) ++score;
		start = 1;
		break;
	    case KEY_RIGHT:
		if (direction == RIGHT) ++score;
		start = 1;
		break;
	    case 'q':
	    case 'Q':
		done = 1;
		break;
	    default:
		break;
	}
      
	if (!done && start) ++loops;
    }

    display_score(score, loops);
}

int main(int argc, char **argv)
{   
    setup();

    play();

    return 0;
}
